IMPLEMENTATION MODULE FIO;

FROM SYSTEM IMPORT
    BYTE, UNREFERENCED_PARAMETER, FUNC, EXITCODE;

FROM SysClock IMPORT
    DateTime, GetClock;

FROM Strings IMPORT
    Equal;

FROM TimeFunc IMPORT
    DateTimeToDos, DosToDateTime;

FROM StdHandles IMPORT
    StdInputHandle, StdOutputHandle, StdErrorHandle;

FROM Str IMPORT
    StrToInt, StrToCard, StrToReal,
    IntToStr, CardToStr, RealToStr, FixRealToStr;

FROM FileFunc IMPORT
    AccessModes, FileAttributes, FileAttributeSet, InvalidHandle,
    FileSpecString, MustHaveDirectory, MustHaveNormalFile, StdAttributes,
    OpenFile, CreateFile, CloseFile, FakeFileOpen, DeleteFile, FileExists,
    ReadBlock, WriteBlock,
    SetFilePos, GetFilePos, MoveFilePos,
    FileLength, SetFileBuffer, FlushBuffers,
    GetFileDateTime, SetFileDateTime, RenameFile, TruncateFile,
    SetDefaultPath, GetDefaultPath, MakeDir, DeleteDir,
    FindFirst, FindNext, FindClose, SearchEntry;
IMPORT FileFunc;

TYPE
    InputString		= ARRAY [0..255] OF CHAR;
    OutputString	= ARRAY [0..79] OF CHAR;

VAR
    Files		: ARRAY [0..MaxOpenFiles] OF FileFunc.File;
    LastResult		: CARDINAL;

PROCEDURE AllocFile() : CARDINAL;
VAR
    i	: CARDINAL;
BEGIN
    FOR i := 5 TO MaxOpenFiles DO
	IF Files[i].handle = InvalidHandle THEN
	    RETURN i;
	END;
    END;
    RETURN 0;
END AllocFile;

PROCEDURE FreeFile(f : CARDINAL);
BEGIN
    Files[f].handle := InvalidHandle;
END FreeFile;

PROCEDURE CheckError(f : CARDINAL; terminate : BOOLEAN) : BOOLEAN;
BEGIN
    LastResult := Files[f].status;
    IF LastResult <> 0 THEN
	OK := FALSE;
	IF IOcheck THEN
	    IF terminate THEN
		HALT(EXITCODE);
	    END;
	END;
	RETURN FALSE;
    END;
    OK := TRUE;
    RETURN TRUE;
END CheckError;

PROCEDURE ValidHandle(f : CARDINAL) : BOOLEAN;
BEGIN
    IF (f >= 1) AND
       (f <= MaxOpenFiles) AND
       (Files[f].handle <> InvalidHandle)
    THEN
	OK := TRUE;
	RETURN TRUE;
    END;
    OK := FALSE;
    LastResult := 6;
    (*!!!
    HALT(EXITCODE);
    *)
    RETURN FALSE;
END ValidHandle;

PROCEDURE Append(name : ARRAY OF CHAR) : File;
VAR
    f	: CARDINAL;
BEGIN
    f := AllocFile();
    IF f <> 0 THEN
	OpenFile(Files[f], name, ReadWriteDenyAll);
	IF CheckError(f, TRUE) THEN
	    SetFilePos(Files[f], FileLength(Files[f]));
	    IF NOT CheckError(f, TRUE) THEN
		CloseFile(Files[f]);
		FreeFile(f);
		f := 0;
	    END;
	ELSE
	    f := 0;
	END;
    END;
    RETURN f;
END Append;

PROCEDURE AssignBuffer(f : File; VAR buf : ARRAY OF BYTE);
BEGIN
    IF ValidHandle(f) THEN
	SetFileBuffer(Files[f], buf);
	FUNC CheckError(f, FALSE);
    END;
END AssignBuffer;

PROCEDURE ChDir(name : ARRAY OF CHAR);
BEGIN
    IF SetDefaultPath(name) THEN
	LastResult := 0;
    ELSE
	LastResult := 3;
	IF IOcheck THEN
	    HALT(EXITCODE);
	END;
    END;
END ChDir;

PROCEDURE Close(f : File);
BEGIN
    IF ValidHandle(f) THEN
	CloseFile(Files[f]);
	FreeFile(f);
	FUNC CheckError(f, TRUE);
    END;
END Close;

PROCEDURE Create(name : ARRAY OF CHAR) : File;
VAR
    f	: CARDINAL;
BEGIN
    f := AllocFile();
    IF f <> 0 THEN
	CreateFile(Files[f], name);
	IF NOT CheckError(f, TRUE) THEN
	    FreeFile(f);
	    f := 0;
	END;
    END;
    RETURN f;
END Create;

PROCEDURE Erase(name : ARRAY OF CHAR);
BEGIN
    IF DeleteFile(name) THEN
	LastResult := 0;
    ELSE
	LastResult := 2;
	IF IOcheck THEN
	    HALT(EXITCODE);
	END;
    END;
END Erase;

PROCEDURE Exists(name : ARRAY OF CHAR) : BOOLEAN;
BEGIN
    RETURN FileExists(name);
END Exists;

PROCEDURE Flush(f : File);
BEGIN
    IF ValidHandle(f) THEN
	FlushBuffers(Files[f], FALSE);
	FUNC CheckError(f, FALSE);
    END;
END Flush;

PROCEDURE GetCurrentDate() : CARDINAL32;
VAR
    dt		: DateTime;
    date	: CARDINAL16;
    time	: CARDINAL16;
BEGIN
    GetClock(dt);
    DateTimeToDos(dt, date, time);
    RETURN VAL(CARDINAL32, date)*65536 + VAL(CARDINAL32, time);
END GetCurrentDate;

PROCEDURE GetDir(drive : SHORTCARD; VAR name : ARRAY OF CHAR);
VAR
    save	: FileSpecString;
    path	: FileSpecString;
BEGIN
    FUNC GetDefaultPath(save);

    IF drive = 0 THEN
	drive := GetDrive()+1;
    END;
    path[0] := CHR(ORD(drive) -1 + ORD('A'));
    path[1] := ':';
    path[2] := 0C;
    FUNC SetDefaultPath(path);
    FUNC GetDefaultPath(path);
    name := path;

    FUNC SetDefaultPath(save);
END GetDir;

PROCEDURE GetDrive() : SHORTCARD;
VAR
    path	: FileSpecString;
BEGIN
    FUNC GetDefaultPath(path);
    RETURN ORD(CAP(path[0])) - ORD('A');
END GetDrive;

PROCEDURE GetFileDate(f : File) : CARDINAL32;
VAR
    date	: CARDINAL16;
    time	: CARDINAL16;
BEGIN
    IF ValidHandle(f) THEN
	GetFileDateTime(Files[f]);
	DateTimeToDos(Files[f].dt, date, time);
	RETURN date;
    END;
    RETURN 0;
END GetFileDate;

PROCEDURE GetPos(f : File) : CARDINAL32;
VAR
    pos		: CARDINAL32;
BEGIN
    IF ValidHandle(f) THEN
	pos := GetFilePos(Files[f]);
	IF CheckError(f, TRUE) THEN
	    RETURN pos;
	END;
    END;
    RETURN MAX(CARDINAL);
END GetPos;

PROCEDURE IOresult() : CARDINAL;
BEGIN
    RETURN LastResult;
END IOresult;

PROCEDURE MkDir(name : ARRAY OF CHAR);
BEGIN
    IF MakeDir(name) THEN
	LastResult := 0;
    ELSE
	LastResult := 3;
	IF IOcheck THEN
	    HALT(EXITCODE);
	END;
    END;
END MkDir;

PROCEDURE Open(name : ARRAY OF CHAR) : File;
VAR
    f	: CARDINAL;
BEGIN
    f := AllocFile();
    IF f <> 0 THEN
	OpenFile(Files[f], name, ReadWriteDenyAll);
	IF NOT CheckError(f, TRUE) THEN
	    f := 0;
	END;
    END;
    RETURN f;
END Open;

PROCEDURE OpenRead(name : ARRAY OF CHAR) : File;
VAR
    f	: CARDINAL;
BEGIN
    f := AllocFile();
    IF f <> 0 THEN
	OpenFile(Files[f], name, ReadOnlyDenyWrite);
	IF NOT CheckError(f, TRUE) THEN
	    f := 0;
	END;
    END;
    RETURN f;
END OpenRead;

PROCEDURE RdBin(f : File;
		VAR buf : ARRAY OF BYTE;
		amount : CARDINAL) : CARDINAL;
BEGIN
    IF ValidHandle(f) THEN
	ReadBlock(Files[f], buf, amount);
	FUNC CheckError(f, FALSE);
	RETURN Files[f].count;
    END;
    RETURN 0;
END RdBin;

PROCEDURE RdItem(f : File; VAR str : ARRAY OF CHAR);
VAR
    ch	: CHAR;
    i	: CARDINAL;
BEGIN
    IF ValidHandle(f) THEN
	LOOP
	    ReadBlock(Files[f], ch, SIZE(ch));
	    IF (Files[f].status = 0) AND
	       (Files[f].count = SIZE(ch))
	    THEN
		IF NOT (ch IN Separators) THEN
		    EXIT;
		ELSIF ch = CHR(26) THEN
		    EOF := TRUE;
		    MoveFilePos(Files[f], -SIZE(ch));
		    EXIT;
		END;
	    ELSE
		EXIT;
	    END;
	END;
	EOF := EOF OR Files[f].eof;

	i := 0;
	IF (Files[f].status = 0) AND NOT EOF THEN
	    str[0] := ch;
	    INC(i);
	    LOOP
		IF i <= HIGH(str) THEN
		    ReadBlock(Files[f], ch, SIZE(ch));
		    IF (Files[f].status = 0) AND
		       (Files[f].count = SIZE(ch))
		    THEN
			IF NOT (ch IN Separators) THEN
			    str[i] := ch;
			    INC(i);
			ELSE
			    EXIT;
			END;
		    ELSE
			EXIT;
		    END;
		ELSE
		    EXIT;
		END;
	    END;
	END;

	EOF := Files[f].eof;
	IF i <= HIGH(str) THEN
	    str[i] := 0C;
	END;
	LastResult := Files[f].status;
    END;
END RdItem;

PROCEDURE RdChar(f : File) : CHAR;
VAR
    ch	: CHAR;
BEGIN
    IF ValidHandle(f) THEN
	ReadBlock(Files[f], ch, SIZE(ch));
	RETURN ch;
    END;
    RETURN 0C;
END RdChar;

PROCEDURE RdBool(f : File) : BOOLEAN;
VAR
    str		: InputString;
BEGIN
    RdItem(f, str);
    RETURN Equal(str, "TRUE");
END RdBool;

PROCEDURE RdShtInt(f : File) : SHORTINT;
VAR
    str		: InputString;
    num		: LONGINT;
BEGIN
    RdItem(f, str);
    num := StrToInt(str, 10, OK);
    OK := OK AND
	  (
	   (num >= MIN(SHORTINT)) AND (num <= MAX(SHORTINT))
	  );
    RETURN num;
END RdShtInt;

PROCEDURE RdInt(f : File) : INTEGER;
VAR
    str		: InputString;
    num		: LONGINT;
BEGIN
    RdItem(f, str);
    num := StrToInt(str, 10, OK);
    OK := OK AND
	  (
	   (num >= MIN(INTEGER)) AND (num <= MAX(INTEGER))
	  );
    RETURN num;
END RdInt;

PROCEDURE RdLngInt(f : File) : LONGINT;
VAR
    str		: InputString;
    num		: LONGINT;
BEGIN
    RdItem(f, str);
    num := StrToInt(str, 10, OK);
    RETURN num;
END RdLngInt;

PROCEDURE RdShtCard(f : File) : SHORTCARD;
VAR
    str		: InputString;
    num		: CARDINAL32;
BEGIN
    RdItem(f, str);
    num := StrToCard(str, 10, OK);
    OK := OK AND
	  (
	   (num >= MIN(SHORTCARD)) AND (num <= MAX(SHORTCARD))
	  );
    RETURN num;
END RdShtCard;

PROCEDURE RdCard(f : File) : CARDINAL;
VAR
    str		: InputString;
    num		: CARDINAL32;
BEGIN
    RdItem(f, str);
    num := StrToCard(str, 10, OK);
    OK := OK AND
	  (
	   (num >= MIN(CARDINAL)) AND (num <= MAX(CARDINAL))
	  );
    RETURN num;
END RdCard;

PROCEDURE RdLngCard(f : File) : CARDINAL32;
VAR
    str		: InputString;
    num		: CARDINAL32;
BEGIN
    RdItem(f, str);
    num := StrToCard(str, 10, OK);
    RETURN num;
END RdLngCard;

PROCEDURE RdShtHex(f : File) : SHORTCARD;
VAR
    str		: InputString;
    num		: CARDINAL32;
BEGIN
    RdItem(f, str);
    num := StrToCard(str, 16, OK);
    OK := OK AND
	  (
	   (num >= MIN(SHORTCARD)) AND (num <= MAX(SHORTCARD))
	  );
    RETURN num;
END RdShtHex;

PROCEDURE RdHex(f : File) : CARDINAL;
VAR
    str		: InputString;
    num		: CARDINAL32;
BEGIN
    RdItem(f, str);
    num := StrToCard(str, 16, OK);
    OK := OK AND
	  (
	   (num >= MIN(CARDINAL)) AND (num <= MAX(CARDINAL))
	  );
    RETURN num;
END RdHex;

PROCEDURE RdLngHex(f : File) : CARDINAL32;
VAR
    str		: InputString;
    num		: CARDINAL32;
BEGIN
    RdItem(f, str);
    num := StrToCard(str, 16, OK);
    RETURN num;
END RdLngHex;

PROCEDURE RdReal(f : File) : REAL;
VAR
    str		: InputString;
    num		: LONGREAL;
BEGIN
    RdItem(f, str);
    num := StrToReal(str, OK);
    OK := OK AND
	  (
	   (num >= MIN(REAL)) AND (num <= MAX(REAL))
	  );
    RETURN num;
END RdReal;

PROCEDURE RdLngReal(f : File) : LONGREAL;
VAR
    str		: InputString;
    num		: LONGREAL;
BEGIN
    RdItem(f, str);
    num := StrToReal(str, OK);
    RETURN num;
END RdLngReal;

PROCEDURE RdStr(f : File; VAR str : ARRAY OF CHAR);
VAR
    ch	: CHAR;
    i	: CARDINAL;
BEGIN
    IF ValidHandle(f) THEN
	i := 0;
	LOOP
	    IF i <= HIGH(str) THEN
		ReadBlock(Files[f], ch, SIZE(ch));
		IF (Files[f].status = 0) AND
		   (Files[f].count = SIZE(ch))
		THEN
		    IF (ch <> CHR(26)) AND (ch <> CHR(13)) THEN
			str[i] := ch;
			INC(i);
		    ELSE
			IF ch = CHR(26) THEN
			    MoveFilePos(Files[f], -SIZE(ch));
			    EOF := TRUE;
			    EXIT;
			END;
		    END;
		ELSE
		    EXIT;
		END;
	    ELSE
		EXIT;
	    END;
	END;

	EOF := Files[f].eof;
	IF i <= HIGH(str) THEN
	    str[i] := 0C;
	END;
    END;
END RdStr;

VAR
    SearchResult	: SearchEntry;

PROCEDURE ReadFirstEntry(path : ARRAY OF CHAR;
			 attr : FileAttr;
			 VAR entry : DirEntry) : BOOLEAN;
VAR
    mayHave	: FileAttributeSet;
    mustHave	: FileAttributeSet;
BEGIN
    (* need to simulalate the way DOS does things *)
    (* since that is how TopSpeed is supposed to work *)

    IF directory IN attr THEN
	mustHave := MustHaveDirectory;
    ELSE
	mustHave := MustHaveNormalFile;
    END;

    mayHave := StdAttributes;
    IF readonly IN attr THEN
	INCL(mustHave, ReadOnly);
    END;
    IF system IN attr THEN
	INCL(mustHave, System);
    END;
    IF hidden IN attr THEN
	INCL(mustHave, Hidden);
    END;
    IF archive IN attr THEN
	INCL(mustHave, Archive);
    END;

    IF FindFirst(path, mayHave, mustHave, SearchResult) THEN
	entry.name := SearchResult.name;
	entry.size := SearchResult.size;
	DateTimeToDos(SearchResult.dt, entry.date, entry.time);
	entry.attr := FileAttr{};
	IF ReadOnly IN SearchResult.attribute THEN
	    INCL(entry.attr, readonly);
	END;
	IF System IN SearchResult.attribute THEN
	    INCL(entry.attr, system);
	END;
	IF Hidden IN SearchResult.attribute THEN
	    INCL(entry.attr, hidden);
	END;
	IF Archive IN SearchResult.attribute THEN
	    INCL(entry.attr, archive);
	END;
	IF Directory IN SearchResult.attribute THEN
	    INCL(entry.attr, directory);
	END;
	RETURN TRUE;
    END;
    RETURN FALSE;
END ReadFirstEntry;

PROCEDURE ReadNextEntry(VAR entry : DirEntry) : BOOLEAN;
BEGIN
    IF FindNext(SearchResult) THEN
	entry.name := SearchResult.name;
	entry.size := SearchResult.size;
	DateTimeToDos(SearchResult.dt, entry.date, entry.time);
	entry.attr := FileAttr{};
	IF ReadOnly IN SearchResult.attribute THEN
	    INCL(entry.attr, readonly);
	END;
	IF System IN SearchResult.attribute THEN
	    INCL(entry.attr, system);
	END;
	IF Hidden IN SearchResult.attribute THEN
	    INCL(entry.attr, hidden);
	END;
	IF Archive IN SearchResult.attribute THEN
	    INCL(entry.attr, archive);
	END;
	IF Directory IN SearchResult.attribute THEN
	    INCL(entry.attr, directory);
	END;
	RETURN TRUE;
    END;
    FindClose(SearchResult);
    RETURN FALSE;
END ReadNextEntry;

PROCEDURE Rename(fromName, toName : ARRAY OF CHAR);
BEGIN
    IF RenameFile(fromName, toName) THEN
	LastResult := 0;
    ELSE
	LastResult := 2;
	IF IOcheck THEN
	    HALT(EXITCODE);
	END;
    END;
END Rename;

PROCEDURE RmDir(name : ARRAY OF CHAR);
BEGIN
    IF DeleteDir(name) THEN
	LastResult := 0;
    ELSE
	LastResult := 3;
	IF IOcheck THEN
	    HALT(EXITCODE);
	END;
    END;
END RmDir;

PROCEDURE Seek(f : File; pos : CARDINAL32);
BEGIN
    IF ValidHandle(f) THEN
	SetFilePos(Files[f], pos);
	FUNC CheckError(f, TRUE)
    END;
END Seek;

PROCEDURE SetDrive(drive : SHORTCARD);
VAR
    path	: ARRAY [0..2] OF CHAR;
BEGIN
    path[0] := CHR(ORD(drive) + ORD('A'));
    path[1] := ':';
    path[2] := 0C;
    FUNC SetDefaultPath(path);
    LastResult := 0;
END SetDrive;

PROCEDURE SetFileDate(f : File; dt : CARDINAL32);
BEGIN
    IF ValidHandle(f) THEN
	DosToDateTime(dt / 65536, dt REM 65536, Files[f].dt);
	SetFileDateTime(Files[f]);
	FUNC CheckError(f, FALSE);
    END;
END SetFileDate;

PROCEDURE Size(f : File) : CARDINAL32;
VAR
    num	: CARDINAL32;
BEGIN
    IF ValidHandle(f) THEN
	num := FileLength(Files[f]);
	IF CheckError(f, TRUE) THEN
	    RETURN num;
	END;
    END;
    RETURN 0;
END Size;

PROCEDURE Truncate(f : File);
BEGIN
    IF ValidHandle(f) THEN
	TruncateFile(Files[f]);
	FUNC CheckError(f, TRUE);
    END;
END Truncate;

PROCEDURE WrBin(f : File; buf : ARRAY OF BYTE; amount : CARDINAL);
BEGIN
    IF ValidHandle(f) THEN
	IF amount <> 0 THEN
	    WriteBlock(Files[f], buf, amount);
	    FUNC CheckError(f, FALSE);
	END;
    END;
END WrBin;

PROCEDURE WrCharRep(f : File; ch : CHAR; amount : CARDINAL);
VAR
    i	: CARDINAL;
BEGIN
    FOR i := 1 TO amount DO
	WrChar(f, ch);
    END;
END WrCharRep;

PROCEDURE WrLn(f : File);
BEGIN
    WrChar(f, CHR(13));
    WrChar(f, CHR(10));
END WrLn;

PROCEDURE WrChar(f : File; ch : CHAR);
BEGIN
    IF ValidHandle(f) THEN
	WriteBlock(Files[f], ch, SIZE(ch));
	OK := (Files[f].status = 0) AND (Files[f].count = SIZE(ch));
    END;
END WrChar;

PROCEDURE WrBool(f : File; bool : BOOLEAN; len : INTEGER);
BEGIN
    IF bool THEN
	WrStrAdj(f, 'TRUE', len);
    ELSE
	WrStrAdj(f, 'FALSE', len);
    END;
END WrBool;

PROCEDURE WrShtInt(f : File; num : SHORTINT; len : INTEGER);
VAR
    str		: OutputString;
BEGIN
    IntToStr(num, str, 10, OK);
    IF OK THEN
	WrStrAdj(f, str, len);
    END;
END WrShtInt;

PROCEDURE WrInt(f : File; num : INTEGER; len : INTEGER);
VAR
    str		: OutputString;
BEGIN
    IntToStr(num, str, 10, OK);
    IF OK THEN
	WrStrAdj(f, str, len);
    END;
END WrInt;

PROCEDURE WrLngInt(f : File; num : LONGINT; len : INTEGER);
VAR
    str		: OutputString;
BEGIN
    IntToStr(num, str, 10, OK);
    IF OK THEN
	WrStrAdj(f, str, len);
    END;
END WrLngInt;

PROCEDURE WrShtCard(f : File; num : SHORTCARD; len : INTEGER);
VAR
    str		: OutputString;
BEGIN
    CardToStr(num, str, 10, OK);
    IF OK THEN
	WrStrAdj(f, str, len);
    END;
END WrShtCard;

PROCEDURE WrCard(f : File; num : CARDINAL; len : INTEGER);
VAR
    str		: OutputString;
BEGIN
    CardToStr(num, str, 10, OK);
    IF OK THEN
	WrStrAdj(f, str, len);
    END;
END WrCard;

PROCEDURE WrLngCard(f : File; num : CARDINAL32; len : INTEGER);
VAR
    str		: OutputString;
BEGIN
    CardToStr(num, str, 10, OK);
    IF OK THEN
	WrStrAdj(f, str, len);
    END;
END WrLngCard;

PROCEDURE WrShtHex(f : File; num : SHORTCARD; len : INTEGER);
VAR
    str		: OutputString;
BEGIN
    CardToStr(num, str, 16, OK);
    IF OK THEN
	WrStrAdj(f, str, len);
    END;
END WrShtHex;

PROCEDURE WrHex(f : File; num : CARDINAL; len : INTEGER);
VAR
    str		: OutputString;
BEGIN
    CardToStr(num, str, 16, OK);
    IF OK THEN
	WrStrAdj(f, str, len);
    END;
END WrHex;

PROCEDURE WrLngHex(f : File; num : CARDINAL32; len : INTEGER);
VAR
    str		: OutputString;
BEGIN
    CardToStr(num, str, 16, OK);
    IF OK THEN
	WrStrAdj(f, str, len);
    END;
END WrLngHex;

PROCEDURE WrReal(f : File;
		 num : REAL;
		 precision : CARDINAL;
		 len : INTEGER);
VAR
    str	: OutputString;
BEGIN
    RealToStr(num, precision, Eng, str, OK);
    IF OK THEN
	WrStrAdj(f, str, len);
    END;
END WrReal;

PROCEDURE WrLngReal(f : File;
		    num : LONGREAL;
		    precision : CARDINAL;
		    len : INTEGER);
VAR
    str	: OutputString;
BEGIN
    RealToStr(num, precision, Eng, str, OK);
    IF OK THEN
	WrStrAdj(f, str, len);
    END;
END WrLngReal;

PROCEDURE WrFixReal(f : File;
		    num : REAL;
		    precision : CARDINAL;
		    len : INTEGER);
VAR
    str	: OutputString;
BEGIN
    FixRealToStr(num, precision, str, OK);
    IF OK THEN
	WrStrAdj(f, str, len);
    END;
END WrFixReal;

PROCEDURE WrFixLngReal(f : File;
		       num : LONGREAL;
		       precision : CARDINAL;
		       len : INTEGER);
VAR
    str	: OutputString;
BEGIN
    FixRealToStr(num, precision, str, OK);
    IF OK THEN
	WrStrAdj(f, str, len);
    END;
END WrFixLngReal;

PROCEDURE WrStr(f : File; str : ARRAY OF CHAR);
BEGIN
    WrBin(f, str, LENGTH(str));
END WrStr;

PROCEDURE WrStrAdj(f : File; str : ARRAY OF CHAR; len : INTEGER);
VAR
    delta	: INTEGER;
    i		: INTEGER;
    ch		: CHAR;
BEGIN
    IF ValidHandle(f) THEN
	delta := ABS(len) - INT(LENGTH(str));

	IF (delta < 0) AND ChopOff THEN
	    ch := '?';
	    FOR i := 1 TO ABS(len) DO
		WriteBlock(Files[f], ch, SIZE(ch));
		IF NOT CheckError(f, FALSE) THEN
		    RETURN;
		END;
	    END;
	    OK := FALSE;
	ELSE
	    OK := TRUE;
	    IF (delta > 0) AND (len > 0) THEN
		FOR i := 1 TO delta DO
		    WriteBlock(Files[f], PrefixChar, SIZE(PrefixChar));
		    IF NOT CheckError(f, FALSE) THEN
			RETURN;
		    END;
		END;
	    END;
	    WriteBlock(Files[f], str, LENGTH(str));
	    IF NOT CheckError(f, FALSE) THEN
		RETURN;
	    END;
	    IF (delta > 0) AND (len < 0) THEN
		FOR i := 1 TO delta DO
		    WriteBlock(Files[f], PostfixChar, SIZE(PostfixChar));
		    IF NOT CheckError(f, FALSE) THEN
			RETURN;
		    END;
		END;
	    END;
	END;
    END;
END WrStrAdj;

PROCEDURE DefaultHandler(addr : CARDINAL32;
			 num : CARDINAL;
			 mess : ARRAY OF CHAR);
BEGIN
    UNREFERENCED_PARAMETER(addr);
    UNREFERENCED_PARAMETER(num);
    UNREFERENCED_PARAMETER(mess);
    HALT(EXITCODE);
END DefaultHandler;

VAR
    i	: CARDINAL;
BEGIN
    IOcheck := TRUE;
    EOF := FALSE;
    Separators := CHARSET{CHR(9), CHR(10), CHR(13), CHR(26), ' '};
    OK := TRUE;
    ChopOff := FALSE;
    Eng := FALSE;
    PrefixChar := ' ';
    PostfixChar := ' ';
    RunTimeError := DefaultHandler;

    FOR i := 0 TO MaxOpenFiles DO
	FakeFileOpen(Files[i], InvalidHandle, ReadWriteDenyNone);
    END;

    FakeFileOpen(Files[StandardInput], StdInputHandle(), ReadOnlyDenyWrite);
    FakeFileOpen(Files[StandardOutput], StdOutputHandle(), WriteOnlyDenyAll);
    FakeFileOpen(Files[ErrorOutput], StdErrorHandle(), WriteOnlyDenyAll);

    %IF DOS %OR FlashTek %THEN
	OpenFile(Files[AuxDevice], "AUX", ReadWriteDenyNone);
	OpenFile(Files[PrinterDevice], "PRN", WriteOnlyDenyNone);
    %END
END FIO.
